<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Support\Facades\DB;

class LogSlowQueries
{
    /**
     * Log DB queries exceeding threshold and add response time header.
     */
    public function handle($request, Closure $next)
    {
        $threshold = (int) (config('app.slow_query_ms', 150));
        $url = $request->fullUrl();
        $userId = optional($request->user())->id;
        $conn = DB::connection()->getName();

        DB::listen(function ($query) use ($threshold, $url, $userId, $conn) {
            try {
                $time = (int) ($query->time ?? 0);
                if ($time >= $threshold) {
                    DB::connection()->table('slow_queries')->insert([
                        'sql' => (string) $query->sql,
                        'bindings' => json_encode($query->bindings, JSON_PARTIAL_OUTPUT_ON_ERROR),
                        'time_ms' => $time,
                        'connection' => $conn,
                        'url' => substr((string) $url, 0, 1024),
                        'user_id' => $userId,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            } catch (\Throwable $e) {
                // Swallow logging errors; never break requests
            }
        });

        $start = microtime(true);
        $response = $next($request);
        try {
            $duration = (int) round((microtime(true) - $start) * 1000);
            $response->headers->set('X-Response-Time', $duration.'ms');
        } catch (\Throwable $e) {
            // ignore header set issues
        }
        return $response;
    }
}

